<?php

function loadCSV($filename) {
    $filepath = __DIR__ . '/csv/' . $filename;
    
    if (!file_exists($filepath)) {
        return ['headers' => [], 'data' => []];
    }
    
    $data = [];
    $headers = [];
    
    if (($handle = fopen($filepath, 'r')) !== FALSE) {
        $headers = fgetcsv($handle, 10000, ',');
        
        while (($row = fgetcsv($handle, 10000, ',')) !== FALSE) {
            if (count($row) === count($headers)) {
                $rowData = [];
                foreach ($headers as $index => $header) {
                    $rowData[$header] = $row[$index];
                }
                $data[] = $rowData;
            }
        }
        fclose($handle);
    }
    
    return ['headers' => $headers, 'data' => $data];
}

function getYearsData() {
    $tvYears = loadCSV('20190308-tv-years.csv');
    $radioYears = loadCSV('20190308-radio-years.csv');
    
    $result = [
        'tv' => [],
        'radio' => [],
        'years' => []
    ];
    
    // Colonnes à exclure (métadonnées)
    $excludeColumns = ['year', 'Médiane', 'Médiane privé', 'Médiane public'];
    
    if (!empty($tvYears['data'])) {
        foreach ($tvYears['data'] as $row) {
            $year = $row['year'];
            $result['years'][] = $year;
            
            // Structurer les données par chaîne
            $tvData = [];
            foreach ($row as $key => $value) {
                if (!in_array($key, $excludeColumns) && !empty($value) && is_numeric($value)) {
                    $tvData[$key] = floatval($value);
                }
            }
            $result['tv'][$year] = $tvData;
        }
    }
    
    if (!empty($radioYears['data'])) {
        foreach ($radioYears['data'] as $row) {
            $year = $row['year'];
            if (!in_array($year, $result['years'])) {
                $result['years'][] = $year;
            }
            
            // Structurer les données par chaîne
            $radioData = [];
            foreach ($row as $key => $value) {
                if (!in_array($key, $excludeColumns) && !empty($value) && is_numeric($value)) {
                    $radioData[$key] = floatval($value);
                }
            }
            $result['radio'][$year] = $radioData;
        }
    }
    
    sort($result['years']); // Trier en ordre croissant
    
    return $result;
}

function getHourlyData() {
    $hourData = loadCSV('20190308-hourstatall.csv');
    
    $result = [];
    
    if (!empty($hourData['data'])) {
        foreach ($hourData['data'] as $row) {
            $channel = $row['channel_name'];
            $year = strval($row['year']);
            $hour = intval($row['hour']);
            
            if (!isset($result[$channel])) {
                $result[$channel] = [];
            }
            
            if (!isset($result[$channel][$year])) {
                $result[$channel][$year] = [];
            }
            
            $result[$channel][$year][$hour] = [
                'women_expression_rate' => floatval($row['women_expression_rate']),
                'speech_rate' => floatval($row['speech_rate']),
                'nb_hours_analyzed' => floatval($row['nb_hours_analyzed']),
                'media_type' => $row['media_type'],
                'is_public' => $row['is_public_channel'] === 'True'
            ];
        }
    }
    
    return $result;
}

function getChannelsData() {
    $channelsData = loadCSV('ina-csa-parole-femmes-chaines.csv');
    
    $result = [];
    
    if (!empty($channelsData['data'])) {
        foreach ($channelsData['data'] as $row) {
            $channelName = $row['Editeur'];
            
            $result[$channelName] = [
                'media_type' => $row['media'],
                'group' => $row['group'],
                'women_2019' => floatval($row['women_expression_rate_2019'] ?? 0) * 100,
                'women_2020' => floatval($row['women_expression_rate_2020'] ?? 0) * 100,
                'speech_rate_2019' => floatval($row['speech_rate_2019'] ?? 0) * 100,
                'speech_rate_2020' => floatval($row['speech_rate_2020'] ?? 0) * 100,
            ];
        }
    }
    
    return $result;
}

function getProgramGenreData() {
    $genreData = loadCSV('ina-csa-parole-femmes-genreprogramme.csv');
    
    $result = [];
    
    if (!empty($genreData['data'])) {
        foreach ($genreData['data'] as $row) {
            $genre = $row['genre'];
            
            if (empty($genre)) continue;
            
            $result[$genre] = [
                'women_2019' => floatval($row['women_expression_rate_2019'] ?? 0) * 100,
                'women_2020' => floatval($row['women_expression_rate_2020'] ?? 0) * 100,
                'speech_rate_2019' => floatval($row['speech_rate_2019'] ?? 0) * 100,
                'speech_rate_2020' => floatval($row['speech_rate_2020'] ?? 0) * 100,
                'nb_declarations_2019' => intval($row['nb_declarations_2019'] ?? 0),
                'nb_declarations_2020' => intval($row['nb_declarations_2020'] ?? 0),
            ];
        }
    }
    
    return $result;
}

function getAllData() {
    return [
        'years' => getYearsData(),
        'hourly' => getHourlyData(),
        'channels' => getChannelsData(),
        'genres' => getProgramGenreData()
    ];
}

?>